<?php
namespace Core;
use Core\Validators\File_Validator;
use Core\File_Handler\Resize_Image;

class File_Uploader{
    use ErrorsTrait;
    private $table_name = "yooz_file_manager";
    private $allowed_file_size = 204800000;//200 MB allowed file size
    private $db;
    private $file = null;
    private $server_name;
    private $ftp_configs;
    private $thumbnail_sizes = ["small_thumbnail"=>150, "medium_thumbnail"=>300, "large_thumbnail"=>450];
    public function __construct($file , $server_name = "local"){
        $this->db = new YoozDB();
        if($this->db->check_connection()){
            if(is_uploaded_file($file['tmp_name'])){
                $validator = new File_Validator();
                if($validator->validate($file)){
                    $validated_file = $validator->get_validated_file();
                    if($this->has_allowed_file_size($validated_file)){
                        $this->ftp_configs = $this->db->get_ftp_configs();
                        $this->set_server_name($server_name);
                        $this->file = $validated_file;
                    }
                }else{
                    $this->set_error($validator->get_errors_text());
                }
            }else{
                $this->set_error("فایل آپلود شده باید توسط روش HTTP POST ارسال شود.");
            }
        }else{
            $this->set_error("ارتباط با دیتابیس برقرار نشد.");
        }
    }
    private function has_allowed_file_size($file){
        $response = (!$this->has_error() && $file['size'] <= $this->allowed_file_size);
        return $this->set_error_on_false($response , "حجم تصویر انتخاب شده باید کمتر از ".$this->get_real_allowed_size()." باشد.");
    }
    private function set_server_name($server_name){
        if($server_name == "local" || array_key_exists($server_name , $this->ftp_configs)){
            $this->server_name = $server_name;
        }else{
            $this->set_error("نام سرور انتخاب شده برای ذخیره فایل نادرست است.");
        }
    }
    private function get_assets_path(){
        return "public/uploads/yooz_uploads/".date("Y")."/".date("m")."/".date("d");
    }
    private function get_real_allowed_size(){
        $size = $this->allowed_file_size;
        switch(true){
            case $size >= 1024:
                return ceil($size / 1024)." مگابایت";
            case $size >= 1024000000:
                return round($size / 1024000000 , 1)." گیگابایت";
            default:
                return $size." کیلوبایت";
        }
    }

    public function uplaod_file() : bool
    {
        if(!$this->has_error() && !empty($this->server_name)){
            if($this->server_name == "local"){
                return $this->local_upload_file();
            }else{
                return $this->ftp_upload_file();
            }
        }
        return false;
    }

    /* =========== upload file into local server =========== */
    private function local_upload_file() : bool
    {
        $response = false;
        if(!$this->has_error() && !empty($this->file) && !empty($this->file['tmp_name'])){
            $server_url = Utils::get_site_url();
            $assetsPathUri = "/".$this->get_assets_path();
            $assetsPath = $_SERVER['DOCUMENT_ROOT'].$assetsPathUri;
            if($this->local_make_dir($assetsPath)){
                $file_upload_name = $assetsPath."/".$this->get_unique_file_name($assetsPathUri , $server_url);
                $file_upload_size = $this->file['size'];
                move_uploaded_file($this->file['tmp_name'], $file_upload_name);
                if(file_exists($file_upload_name)){
                    $response = true;
                    chmod($file_upload_name , 0755);
                    $thumbnails = $this->local_create_thumbnails($file_upload_name);
                    $file_url = $server_url.str_replace($_SERVER['DOCUMENT_ROOT'] , '' , $file_upload_name);
                    $fileData = array();
                    $fileData["file_size"] = $file_upload_size;
                    $imageSizes = @getimagesize($file_upload_name);
                    if($imageSizes){
                        $fileData['image_width'] = $imageSizes[0];
                        $fileData['image_height'] = $imageSizes[1];
                    }
                    if(!empty($thumbnails) && is_array($thumbnails)){
                        foreach ($thumbnails as $thumbnailName => $thumbnailData){
                            $fileData[$thumbnailName] = $thumbnailData['url'];
                        }
                    }
                    $this->seve_file_into_db($file_url , $server_url , $fileData);
                }else{
                    $this->set_error("آپلود فایل با شکست مواجه شد.");
                }
            }else{
                $this->set_error("ساخت دایرکتوری برای ذخیره فایل با شکست مواجه شد.");
            }
        }
        return $response;
    }
    private function local_make_dir($dir){
        if(!file_exists($dir)){
            return mkdir($dir , 0755, true);//if true is set , if directory not exists , mkdir create whole it
        }else{
            return true;
        }
    }


    /* =========== upload file into ftp server =========== */
    private function ftp_upload_file(): bool
    {
        $response = false;
        if(!$this->has_error() && !empty($this->file) && !empty($this->file['tmp_name'])) {
            if(is_array($this->ftp_configs) && !empty($this->ftp_configs[$this->server_name])) {
                $ftpInfo = $this->ftp_configs[$this->server_name];
                $ftpTimeout = !empty($ftpInfo['timeout']) ? $ftpInfo['timeout'] : 90;
                $ftpConnect = $this->connect_to_ftp($ftpInfo['server'] , $ftpInfo['port'] , $ftpTimeout);
                if($ftpConnect){
                    $ftpLogin = $this->login_to_ftp($ftpConnect , $ftpInfo);
                    @ftp_pasv($ftpConnect , true);
                    if($ftpLogin){
                        $server_url = $ftpInfo['server_url'];
                        $assetsPath = "/".$this->get_assets_path();
                        if($this->ftp_mksubdirs($ftpConnect, $ftpInfo['home_directory'], $assetsPath)){
                            $remote_file = $ftpInfo['home_directory'].$assetsPath."/".$this->get_unique_file_name($assetsPath , $server_url);
                            $remote_file_size = $this->file['size'];
                            if(@ftp_put($ftpConnect , $remote_file , $this->file['tmp_name'] , FTP_BINARY)){
                                $response = true;
                                $thumbnail_destination_dir = $ftpInfo['home_directory'].$assetsPath;
                                $thumbnail_destination_url = $server_url.$assetsPath;
                                $file_url = str_replace($ftpInfo['home_directory'], $server_url ,$remote_file);
                                $ftpThumbnails = $this->ftp_create_thumbnails($ftpConnect , $file_url , $thumbnail_destination_dir , $thumbnail_destination_url);

                                /* prepare data to save in database */
                                $fileData = array();
                                $fileData["file_size"] = $remote_file_size;
                                $imageSizes = @getimagesize(str_replace(" " , "%20", $file_url));
                                if($imageSizes){
                                    $fileData['image_width'] = $imageSizes[0];
                                    $fileData['image_height'] = $imageSizes[1];
                                }
                                if(!empty($ftpThumbnails) && is_array($ftpThumbnails)){
                                    foreach ($ftpThumbnails as $thumbnailName => $thumbnailData){
                                        $fileData[$thumbnailName] = $thumbnailData['url'];
                                    }
                                }
                                $this->seve_file_into_db($file_url, $server_url ,$fileData);
                            }else{
                                $this->set_error("بارگذاری فایل روی Remote Server ناموفق بود.");
                            }
                        }
                    }
                }else{
                    $this->set_error("اتصال با ftp برقرار نشد.");
                }
            }else{
                $this->set_error("اطلاعات Remote Server انتخاب شده، یافت نشد.");
            }
        }
        return $response;
    }
    private function connect_to_ftp($server="", int $port=21, int $timeout=90){
        $response = false;
        if(!$this->has_error() && $this->server_name !== "local" && !empty($server)){
            $connect = @ftp_connect($server , $port , $timeout);
            if($connect){
                $response = $connect;
            }
        }
        return $response;
    }
    private function login_to_ftp($ftpConnect , array $ftpInfo){
        $response = false;
        if(!$this->has_error() && $this->server_name !== "local" && !empty($ftpInfo)){
            $ftpUrl = $ftpInfo['server_url'];
            if(filter_var($ftpUrl, FILTER_VALIDATE_URL)){
                $ftpLogin = @ftp_login($ftpConnect , $ftpInfo['username'] , $ftpInfo['password']);
                if($ftpLogin){
                    $response = $ftpLogin;
                }else{
                    $this->set_error("ورود به ftp ناموفق بود.");
                }
            }else{
                $this->set_error("نشانی اینترنتی سرور قابل قبول نیست.");
            }
        }
        return $response;
    }
    private function ftp_create_thumbnails($ftp_connect , $source_file , $destination_dir , $destination_url){
        $response = array();
        if(!$this->has_error()){
            $fileName = pathinfo($source_file , PATHINFO_FILENAME);
            $fileType = strtolower(pathinfo($source_file , PATHINFO_EXTENSION));
            if($fileType == "jpeg"){$fileType = "jpg";}
            if($fileType == "jpg" || $fileType == "png"){
                $custom_dest_path = $_SERVER['DOCUMENT_ROOT']."/public/uploads/yooz_uploads";
                $custom_file_name = "_FTP_TMP_THUMBNAIL_FILE";
                $thumbnails = $this->local_create_thumbnails($source_file, $custom_dest_path,$custom_file_name , true);
                if($thumbnails && is_array($thumbnails)){
                    foreach($thumbnails as $thumbnailName => $thumbnailData){
                        $TMP_Thumbnail = $thumbnailData['file'];
                        $thumbnail_file_name = pathinfo(str_replace($custom_file_name , $fileName , $TMP_Thumbnail) , PATHINFO_BASENAME);
                        $remote_thumbnail = $destination_dir."/".$thumbnail_file_name;
                        $remote_thumbnail_url = $destination_url."/".$thumbnail_file_name;
                        if(@ftp_put($ftp_connect , $remote_thumbnail , $TMP_Thumbnail , FTP_BINARY) == false){
                            $this->set_error("آپلود تصاویر کوچک با مشکل مواجه شد.");
                        }else{
                            $response[$thumbnailName] = array("file"=>$remote_thumbnail, "url"=>$remote_thumbnail_url);
                        }
                        unlink($TMP_Thumbnail);
                    }
                }
            }
        }
        return $response;
    }
    private function ftp_mksubdirs($connect , string $home_dir , string $path){
        $ch_home = @ftp_chdir($connect, $home_dir);
        if($ch_home){
            $currentPath = $home_dir;
            $subDirs = explode('/' , $path);
            foreach($subDirs as $sub_dir){
                if(!empty($sub_dir)){
                    if(@ftp_chdir($connect, $currentPath."/".$sub_dir) == false){
                        ftp_mkdir($connect, $sub_dir);
                    }
                    //try again after trying to create sub directory
                    if(@ftp_chdir($connect, $currentPath."/".$sub_dir)){
                        $currentPath = $currentPath."/".$sub_dir;
                    }else{
                        $this->set_error("ساخت دایرکتوری برای ذخیره فایل در Remote Server با شکست مواجه شد.");
                    }
                }
            }
            return @ftp_chdir($connect, $home_dir.$path);
        }else{
            $this->set_error_on_false(false ,"آدرس دایرکتوری Home نادرست است. لطفا در فایل کانفیگ مقدار آن را اصلاح نمایید.");
        }
    }



    private function get_unique_file_name($file_path , $server_url){
        $uniqueName = "";
        if(!$this->has_error() && !empty($this->file) && !empty($this->file['name'])){
            $file_name = pathinfo($this->file['name'] , PATHINFO_FILENAME);
            $file_ext = pathinfo($this->file['name'] , PATHINFO_EXTENSION);
            $uniqueName = $file_name.".".$file_ext;
            $nameStmt = $this->db->prepare("SELECT file_name FROM yooz_file_manager WHERE file_name=:fileName AND server_url=:serverUrl AND file_path=:filePath");
            $nameStmt->bindParam(":fileName" , $uniqueName);
            $nameStmt->bindParam(":serverUrl" , $server_url);
            $nameStmt->bindParam(":filePath" , $file_path);
            $isUnique = false;
            $i = 0;
            while($isUnique == false){
                $i++;
                $nameStmt->execute();
                if($nameStmt->rowCount() == 0){
                    $isUnique = true;
                }else{
                    $uniqueName = $file_name." (".$i.").".$file_ext;
                }
            }
            $nameStmt->closeCursor();
        }
        return $uniqueName;
    }
    private function local_create_thumbnails($source_file , $custom_dest_path="" ,$custom_file_name="" , bool $encode_source = false){
        $thumbnails = [];
        if(!empty($this->thumbnail_sizes) && is_array($this->thumbnail_sizes)){
            if(!empty($source_file)){
                $assetsPath = $_SERVER['DOCUMENT_ROOT']."/".$this->get_assets_path();
                if($custom_dest_path){
                    $assetsPath = $custom_dest_path;
                }
                $fileType = strtolower(pathinfo($source_file , PATHINFO_EXTENSION));
                if($fileType === "jpeg"){$fileType = "jpg";}
                $fileName = pathinfo($source_file , PATHINFO_FILENAME);
                if(!empty($custom_file_name)){
                    $fileName = $custom_file_name;
                }
                if($fileType == "png" || $fileType == "jpg"){
                    $resizer = new Resize_Image();
                    foreach($this->thumbnail_sizes as $size_name => $size){
                        $thumbnailName = $assetsPath."/".$fileName."-".$size."px.".$fileType;
                        if($resizer->create_resized_image($thumbnailName , $source_file , $size, $size , false , true, $encode_source)){
                            $thumbnail_url = Utils::get_site_url()."/".str_replace($_SERVER['DOCUMENT_ROOT'] , '' , $thumbnailName);
                            $thumbnails[$size_name] = array("file"=>$thumbnailName,"url"=>$thumbnail_url);
                        }else{
                            $this->set_error("ساخت تصاویر کوچک ناموفق بود.");
                            break;
                        }
                    }
                }
            }
        }
        return $thumbnails;
    }

    protected $filesExtType = array(
        "jpg"   => "image",
        "png"   => "image",
        "gif"   => "image",
        "webp"  => "image",
        "svg"   => "image",
        "mp3"   => "audio",
        "m4a"   => "audio",
        "mp4"   => "video",
        "zip"   => "zip",
        "pdf"   => "pdf",
    );
    private function seve_file_into_db($file_url , $server_url , $file_data){
        $data = [];
        $data['file_url'] = $file_url;
        $data['server_url'] = $server_url;
        $data['file_title'] = pathinfo($file_url , PATHINFO_FILENAME);
        $data['file_name'] = pathinfo($file_url , PATHINFO_BASENAME);
        $file_type = strtolower(pathinfo($file_url , PATHINFO_EXTENSION));
        $data['file_type'] = !empty($this->filesExtType[$file_type]) ? $this->filesExtType[$file_type] : "unknown";
        $file_dir = pathinfo($file_url , PATHINFO_DIRNAME);
        $data['file_path'] = str_replace($server_url , "", $file_dir);
        $data['file_data'] = serialize($file_data);

        $saveResult = $this->db->insert($this->table_name , $data);
        if(!$saveResult){
            $this->set_error("ذخیره اطلاعات فایل بارگذاری شده در دیتابیس با شکست مواجه شد.");
        }
    }

    public function add_thumbnail_size(string $key , int $size){
        if(!array_key_exists($key , $this->thumbnail_sizes)){
            $this->thumbnail_sizes[$key] = $size;
        }
    }

    public function set_allowed_file_size(int $new_file_size = 0){
        if(!empty($new_file_size) && is_numeric($new_file_size)){
            $this->allowed_file_size = intval($new_file_size);
        }
    }

}
?>